' SAP DPI Document Access Automation Script
' Author: Dario Pascoal
'
' Description: Automated DPI document discovery and access system that integrates 
' with SAP GUI to locate and open DPI (Detailed Part Inspection) documents through 
' the ZPACK transaction. This script handles the complete workflow from material 
' number validation to document location, selection, and viewer application launch 
' for inspection document review and analysis.
'
' The script automates the complex process of navigating SAP's document management 
' system, searching for DPI documents associated with specific materials, and 
' launching the appropriate document viewer for inspection procedures. It includes 
' comprehensive error handling, logging capabilities, and security configuration 
' management to ensure reliable unattended operation.
'
' Functionality:
' - Connects to active SAP GUI session for automated interaction
' - Validates and processes material numbers for document searches
' - Locates DPI documents in SAP's ZPACK transaction system
' - Launches document viewers for inspection document access
' - Provides detailed logging for audit trails and troubleshooting
' - Handles SAP security dialogs and authentication automatically
' - Maintains SAP session state for continued user interaction
'
' Prerequisites:
' - Active SAP GUI session with user authentication
' - SAP GUI Scripting enabled and properly configured
' - Access to ZPACK transaction for document management operations
' - Windows Script Host (WSH) environment available
' - Appropriate SAP user permissions for document access
' - Document viewer applications installed (Adobe, PowerPoint, Edge)
'
' Parameters:
' - Arg 0: Material Number (string) - Equipment/material identifier for document search
' - Arg 1: Log File Path (optional string) - Path for operation logging
'
' Returns:
' - Console output with operation status and completion messages
' - Log file entries with detailed operation tracking (if path provided)
' - Document viewer application launched with DPI document (if found)
' - SAP GUI remains on document list for continued user interaction
'
' Error Handling:
' - Comprehensive error catching for SAP connection issues
' - Graceful handling of security dialog automation
' - Log file creation and access error management
' - Registry modification error handling for SAP scripting
' - Document not found scenarios with appropriate status reporting
'
' Maintenance Notes:
' - DPI document identification uses case-insensitive "DPI" string matching
' - SAP GUI element IDs may change with SAP system updates
' - Security registry modifications require appropriate user permissions
' - Document viewer selection depends on file type and installed applications

' =============================================================================
' ENVIRONMENT INITIALIZATION AND ERROR HANDLING
' =============================================================================
' Configure script execution environment with comprehensive error handling
' and security settings for reliable SAP GUI automation

' Enable comprehensive error handling for graceful failure management
' This allows the script to continue operation even when non-critical errors occur
On Error Resume Next

' =============================================================================
' WINDOWS SHELL AND REGISTRY CONFIGURATION
' =============================================================================
' Initialize Windows Shell object for system-level operations and SAP GUI
' scripting security configuration through registry modifications

' Create Windows Shell object for system interaction and registry access
' This object provides access to Windows registry and system automation features
Set WSHShell = CreateObject("WScript.Shell")

' =============================================================================
' SAP GUI SECURITY CONFIGURATION
' =============================================================================
' Disable SAP GUI scripting security warnings to enable automated interaction
' These registry modifications are essential for unattended script execution
' and prevent user prompts that would interrupt the automation workflow

' Disable attachment warnings when scripts connect to SAP GUI sessions
' This prevents popup dialogs that would halt the DPI document access automation
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAttach", 0, "REG_DWORD"

' Disable connection warnings for script-to-SAP communication
' This ensures seamless connection establishment without user intervention
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnConnection", 0, "REG_DWORD"

' Disable allow list requirement warnings for automated scripting
' This prevents security prompts that would require manual user interaction
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\WarnOnAllowListRequired", 0, "REG_DWORD"

' Disable allow list usage to prevent script blocking
' This allows our automation script to execute without whitelist restrictions
WSHShell.RegWrite "HKEY_CURRENT_USER\Software\SAP\SAPGUI Front\SAP Frontend Server\Security\UseAllowList", 0, "REG_DWORD"

' =============================================================================
' COMMAND LINE ARGUMENT PROCESSING
' =============================================================================
' Extract and validate command line arguments for material number processing
' and optional log file configuration for detailed operation tracking

' Extract material number from first command line argument (required)
' This material number identifies the equipment/material for DPI document search
materialNumber = WScript.Arguments.Item(0)

' =============================================================================
' LOGGING SYSTEM INITIALIZATION
' =============================================================================
' Setup comprehensive logging system for operation tracking and debugging
' The logging system supports detailed file logging with timestamps for
' audit trails, troubleshooting, and operational monitoring

' Declare log file objects and path variables for file-based logging
' These objects handle all file-based logging operations throughout the script
Dim logFile, logFilePath

' =============================================================================
' OPTIONAL LOG FILE INITIALIZATION
' =============================================================================
' Initialize file-based logging system if log file path is provided as
' second command line argument for detailed operation tracking

' Initialize log file if path is provided as optional second argument
' This enables detailed file-based logging for audit trails and debugging
If WScript.Arguments.Count > 1 Then
    logFilePath = WScript.Arguments.Item(1)
    
    ' Attempt to open the log file in append mode for continuous logging
    ' Append mode (8) allows multiple script executions to write to same log
    On Error Resume Next
    Set fso = CreateObject("Scripting.FileSystemObject")
    Set logFile = fso.OpenTextFile(logFilePath, 8, True)
    
    ' Handle log file creation or access errors gracefully
    ' If log file cannot be opened, continue operation without file logging
    If Err.Number <> 0 Then
        WScript.Echo "ERROR: Could not open log file: " & Err.Description
        Err.Clear
        Set logFile = Nothing
    Else
        ' Write script initialization header to log for session identification
        ' This creates a clear audit trail for each script execution
        LogToFile "========================================="
        LogToFile "Script Started at " & Now
        LogToFile "Script: " & WScript.ScriptName
        LogToFile "Material Number: " & materialNumber
        LogToFile "========================================="
    End If
End If

' =============================================================================
' LOGGING UTILITY FUNCTIONS
' =============================================================================
' Comprehensive logging functions that support both console output and
' file-based logging for complete operation visibility and audit trails

'
' Function to write timestamped messages to log file for audit trail maintenance
' This function provides detailed file-based logging with timestamps for
' troubleshooting, compliance, and operational monitoring purposes
'
' Parameters:
' - message: String message to be logged with timestamp and script identification
'
Sub LogToFile(message)
    On Error Resume Next
    ' Check if log file object is available and properly initialized
    If Not logFile Is Nothing Then
        ' Write formatted log entry with timestamp and script identification
        ' Format: [Timestamp] - [Script Name] - [Message]
        logFile.WriteLine Now & " - [" & WScript.ScriptName & "] - " & message
    End If
End Sub

'
' Function to output messages to both console and log file for comprehensive logging
' This dual-output approach ensures immediate visibility while maintaining
' permanent audit trails for compliance and troubleshooting purposes
'
' Parameters:
' - message: String message to be displayed and logged
'
Sub LogMessage(message)
    ' Display message immediately in console for real-time monitoring
    WScript.Echo message
    
    ' Also write message to log file for permanent audit trail
    LogToFile message
End Sub

' =============================================================================
' SAP GUI CONNECTION ESTABLISHMENT
' =============================================================================
' Establish connection to SAP GUI application and active session for
' automated interaction with SAP's document management functionality

' Initialize SAP GUI application object if not already connected
' The application object is the top-level SAP GUI automation interface
If Not IsObject(application) Then
   ' Connect to the SAP GUI Automation Server for script access
   ' This server manages all SAP GUI automation and scripting operations
   Set SapGuiAuto = GetObject("SAPGUI")
   
   ' Get the scripting engine from the SAP GUI server
   ' The scripting engine provides programmatic access to SAP GUI elements
   Set application = SapGuiAuto.GetScriptingEngine
End If

' Initialize SAP connection object if not already established
' The connection object represents a specific SAP system connection
If Not IsObject(connection) Then
   ' Get the first (primary) connection from available connections
   ' This assumes the user has an active SAP connection established
   Set connection = application.Children(0)
End If

' Initialize SAP session object for transaction execution
' The session object represents the active user session for automation
If Not IsObject(session) Then
   ' Get the first (active) session from the established connection
   ' This session will be used for all SAP transaction automation
   Set session = connection.Children(0)
End If

' Enable event handling for SAP GUI automation if Windows Script Host is available
' This allows the script to respond to SAP GUI events and maintain connection state
If IsObject(WScript) Then
   ' Connect session object to handle SAP GUI session events
   WScript.ConnectObject session, "on"
   
   ' Connect application object to handle SAP GUI application events
   WScript.ConnectObject application, "on"
End If

' =============================================================================
' SECURITY DIALOG AUTOMATION
' =============================================================================
' Automated handling of SAP GUI security dialogs that may appear during
' script execution, ensuring unattended operation without user intervention

'
' Handle security dialog automation for unattended script execution
' This function automatically dismisses security warnings that could
' interrupt the DPI document access workflow and require manual intervention
'
Sub HandleSecurityDialog()
    Dim shell, timeout, dialogFound
    Set shell = CreateObject("WScript.Shell")
    timeout = 0
    dialogFound = False
    
    ' Wait up to 10 seconds for security dialog to appear
    ' This prevents infinite waiting while allowing time for dialog display
    Do While timeout < 10
        ' Check if SAP GUI Security dialog is active
        ' AppActivate returns True if the specified window title is found
        If shell.AppActivate("SAP GUI Security") Then
            ' Brief pause to ensure dialog is fully loaded and ready for input
            ' This prevents sending keys before dialog elements are interactive
            WScript.Sleep 500
            
            ' Send space key to check "Remember my decision" checkbox
            ' This prevents future security dialogs for this script execution
            shell.SendKeys " "
            WScript.Sleep 200  ' Brief pause between key operations
            
            ' Send Enter key to click "Allow" button and dismiss dialog
            ' This grants permission for script to interact with SAP GUI
            shell.SendKeys "{ENTER}"
            dialogFound = True
            Exit Do
        End If
        
        ' Wait 500ms before checking again for dialog appearance
        ' This provides reasonable response time without excessive CPU usage
        WScript.Sleep 500
        timeout = timeout + 1
    Loop
End Sub

' =============================================================================
' MAIN DPI DOCUMENT ACCESS WORKFLOW
' =============================================================================
' Execute the complete DPI document discovery and access workflow through
' SAP's ZPACK transaction including material search and document selection

' Execute security dialog handling before starting SAP operations
' This ensures any security prompts are automatically handled without user intervention
HandleSecurityDialog()

' =============================================================================
' SAP TRANSACTION EXECUTION AND MATERIAL SEARCH
' =============================================================================
' Navigate to ZPACK transaction and execute material search to locate
' DPI documents associated with the specified material number

' Maximize SAP window for optimal visibility and element access
' This ensures all GUI elements are visible and accessible for automation
session.findById("wnd[0]").maximize

' Navigate to ZPACK transaction for material document management
' ZPACK provides access to material-related documents including DPI files
session.findById("wnd[0]/tbar[0]/okcd").text = "zpack"
session.findById("wnd[0]").sendVKey 0

' Enter material number in search field and execute search
' This populates the document table with all available documents for the material
session.findById("wnd[0]/usr/ctxtS_MATNR-LOW").text = materialNumber
session.findById("wnd[0]").sendVKey 8

' =============================================================================
' DOCUMENT TABLE ACCESS AND DPI SEARCH
' =============================================================================
' Access the document list table and search for DPI documents using
' case-insensitive string matching for reliable document identification

' Get reference to the file list table for document enumeration
' This table contains all documents associated with the material number
Set fileTable = session.findById("wnd[0]/usr/tabsTAB_9100/tabpTAB_9100_FC3/ssubTAB_9100_SCA:ZDGP_MATERIAL_OVERVIEW:9103/cntlCONTROL_9103_2/shellcont/shell")

' Initialize DPI document discovery variables
' These track whether DPI documents are found and their table positions
foundDPI = False
dpiRowIndex = -1
foundFileName = ""

' =============================================================================
' DPI DOCUMENT IDENTIFICATION LOOP
' =============================================================================
' Systematically search through all documents in the table to identify
' DPI documents using case-insensitive string matching

' Loop through all documents in the table to find DPI files
' This systematic search ensures all DPI documents are identified
For i = 0 To fileTable.RowCount - 1
    ' Extract document name from the document number column
    ' DOKNR column contains the document identifier and filename information
    fileName = fileTable.GetCellValue(i, "DOKNR")
    
    ' Check for DPI identifier using case-insensitive comparison
    ' This ensures DPI documents are found regardless of case variations
    If InStr(1, fileName, "DPI", vbTextCompare) > 0 Then
        foundDPI = True
        dpiRowIndex = i
        foundFileName = fileName
        ' Exit loop immediately when first DPI document is found
        ' This provides efficient processing for single DPI document access
        Exit For
    End If
Next

' =============================================================================
' DOCUMENT SELECTION AND VIEWER LAUNCH
' =============================================================================
' Execute document selection and launch appropriate viewer application
' based on DPI document availability and successful document location

' Process results and execute appropriate action based on document availability
' This section handles both successful document location and not-found scenarios
If foundDPI Then
    ' ==========================================================================
    ' DPI DOCUMENT FOUND - EXECUTE SELECTION AND LAUNCH
    ' ==========================================================================
    ' DPI document successfully located - proceed with selection and viewer launch
    
    ' Select the DPI document row in the table for activation
    ' This focuses the table on the specific DPI document for processing
    fileTable.setCurrentCell dpiRowIndex, "DOKNR"
    fileTable.selectedRows = dpiRowIndex
    
    ' Double-click the selected document to launch the viewer application
    ' This triggers SAP's document viewer integration to open the DPI file
    fileTable.doubleClickCurrentCell
    
    ' Wait for the document viewer application to fully load
    ' This delay ensures the viewer application has sufficient time to initialize
    ' and load the DPI document before the script continues or terminates
    WScript.Sleep 3000
    
    ' Log successful document access with filename for audit trail
    ' This provides confirmation that the DPI document was successfully located and opened
    LogMessage "SUCCESS: " & foundFileName
    
    ' NOTE: SAP session intentionally remains on document list screen
    ' This design choice allows users to continue browsing documents or
    ' access additional files without requiring navigation back to the list
    
Else
    ' ==========================================================================
    ' DPI DOCUMENT NOT FOUND - LOG STATUS
    ' ==========================================================================
    ' No DPI documents found for the specified material number
    
    ' Log not-found status for troubleshooting and audit purposes
    ' This indicates either no DPI documents exist for the material or
    ' the document naming convention doesn't match the expected "DPI" pattern
    LogMessage "NOT-FOUND"
End If

' =============================================================================
' CLEANUP AND FINALIZATION PROCEDURES
' =============================================================================
' Comprehensive cleanup procedures to ensure proper resource management
' and logging finalization before script termination

'
' Cleanup function to properly close log file and release system resources
' This function ensures proper log file closure and resource cleanup to
' prevent memory leaks and ensure all log data is properly written to disk
'
Sub CloseLogFile()
    On Error Resume Next
    ' Check if log file is still open and available for closure
    If Not logFile Is Nothing Then
        ' Write session end marker to log for clear audit trail separation
        LogToFile "Script ended at " & Now
        LogToFile "========================================="
        
        ' Properly close the log file to ensure all buffered data is written
        logFile.Close
        
        ' Release the log file object to free system resources
        Set logFile = Nothing
    End If
End Sub

' Execute final cleanup procedures before script termination
' This ensures proper resource cleanup and log file closure in all cases
CloseLogFile()